<?php
header('Content-Type: application/json; charset=utf-8');

require __DIR__ . '/../includes/bootstrap.php';
require __DIR__ . '/../includes/formatter.php';
require __DIR__ . '/../includes/rate_limit.php';
require __DIR__ . '/../includes/ai_openrouter.php';
require __DIR__ . '/../includes/hybrid.php';

$debug = (($_GET['debug'] ?? '') === '1');

try {
  enforce_rate_limit($pdo, $app);

  $in = json_decode(file_get_contents('php://input'), true);
  $q  = trim((string)($in['question'] ?? $_GET['q'] ?? ''));

  if ($q === '') {
    echo json_encode(['answer'=>format_rich(
      $app['app_name'],
      'اسم شرکت، حوزه فعالیت یا شماره غرفه رو بنویس.',
      ['مثال: شرکت ملی حفاری', 'مثال: شرکت‌های مرتبط با حفاری', 'مثال: ساعت بازدید نمایشگاه']
    )], JSON_UNESCAPED_UNICODE);
    exit;
  }

  $intent = 'general';

  function is_list_intent($q) {
    $q2 = $q;
    if (function_exists('mb_strtolower')) $q2 = mb_strtolower($q2);
    else $q2 = strtolower($q2);

    $hasCompanyWord = (strpos($q2,'شرکت') !== false);

    // plural/list phrasing (very common): "شرکت‌های ..." / "شرکت ها ..." / "شرکتها ..."
    $plural = (strpos($q2,'شرکت‌های') !== false) ||
              (strpos($q2,'شرکت ها') !== false) ||
              (strpos($q2,'شرکت‌ها') !== false) ||
              (strpos($q2,'شرکتهای') !== false) ||
              (strpos($q2,'شرکت های') !== false);

    $hasListWords = (strpos($q2,'لیست') !== false) ||
                    (strpos($q2,'معرفی') !== false) ||
                    (strpos($q2,'مرتبط') !== false) ||
                    (strpos($q2,'کدام') !== false) ||
                    (strpos($q2,'کدوم') !== false);

    // advice/strategy intent: do NOT route to tag listing
    $advice = (strpos($q2,'بهترین') !== false) ||
              (strpos($q2,'روش') !== false) ||
              (strpos($q2,'چطور') !== false) ||
              (strpos($q2,'چگونه') !== false) ||
              (strpos($q2,'استراتژی') !== false) ||
              (strpos($q2,'بودجه') !== false) ||
              (strpos($q2,'بازاریابی') !== false) ||
              (strpos($q2,'تبلیغ') !== false) ||
              (strpos($q2,'تبلیغات') !== false);

    if ($advice) return false;

    // If user explicitly used plural "companies" form, treat as list intent
    if ($plural) return true;

    return ($hasCompanyWord && $hasListWords);
}

function is_advice_intent($q) {
    $q2 = $q;
    if (function_exists('mb_strtolower')) $q2 = mb_strtolower($q2);
    else $q2 = strtolower($q2);

    return (strpos($q2,'بهترین') !== false) ||
           (strpos($q2,'روش') !== false) ||
           (strpos($q2,'چطور') !== false) ||
           (strpos($q2,'چگونه') !== false) ||
           (strpos($q2,'استراتژی') !== false) ||
           (strpos($q2,'بودجه') !== false) ||
           (strpos($q2,'بازاریابی') !== false) ||
           (strpos($q2,'تبلیغ') !== false) ||
           (strpos($q2,'تبلیغات') !== false) ||
           (strpos($q2,'پیشنهاد') !== false) ||
           (strpos($q2,'ایده') !== false) ||
           (strpos($q2,'سناریو') !== false) ||
           (strpos($q2,'متن') !== false) ||
           (strpos($q2,'کپشن') !== false);
}

  // 1) FAQ
  $faq = match_faq($pdo, $q);
  if ($faq) {
    log_answer($pdo, $q, 'faq', 'faq', null);
    echo json_encode([
      'answer' => format_rich(
        'پاسخ سریع',
        'این مورد داخل FAQ ثبت شده.',
        [trim((string)$faq['answer'])],
        ''
      ),
      'meta' => ['source'=>'faq']
    ], JSON_UNESCAPED_UNICODE);
    exit;
  }

  // 2) Company entity in question
  $company = detect_company_in_question($pdo, $q);
  if ($company) {
    $cid = (int)$company['id'];
    if (!boost_active($company)) $company['boost_enabled'] = 0;

    $related = related_companies($pdo, $cid, 5);
    [$title,$summary,$bullets,$extra] = build_company_answer($company, $related);

    // If sparse, add site links by company name
    if (count($bullets) < 2) {
      $hits2 = search_site($pdo, (string)$company['name'], 3);
      if ($hits2) {
        [,,$sbul,] = build_site_answer($hits2);
        $extra .= ($extra? "\n\n" : "") . "صفحات مرتبط در سایت:\n" . implode("\n", array_map(fn($b)=>"- ".$b, $sbul));
      }
    }

    log_answer($pdo, $q, 'company_entity', 'company', $cid);
    echo json_encode([
      'answer' => format_rich($title, $summary, $bullets, $extra),
      'meta'  => ['source'=>'company','company_id'=>$cid]
    ], JSON_UNESCAPED_UNICODE);
    exit;
  }

  // 3) Companies by LIKE
  $companies = find_companies($pdo, $q);
  if ($companies) {
    $best = $companies[0];
    $cid = (int)$best['id'];
    if (!boost_active($best)) $best['boost_enabled'] = 0;

    $related = related_companies($pdo, $cid, 5);
    [$title,$summary,$bullets,$extra] = build_company_answer($best, $related);

    if (count($bullets) < 2) {
      $hits2 = search_site($pdo, (string)$best['name'], 3);
      if ($hits2) {
        [,,$sbul,] = build_site_answer($hits2);
        $extra .= ($extra? "\n\n" : "") . "صفحات مرتبط در سایت:\n" . implode("\n", array_map(fn($b)=>"- ".$b, $sbul));
      }
    }

    log_answer($pdo, $q, 'company', 'company', $cid);
    echo json_encode([
      'answer' => format_rich($title, $summary, $bullets, $extra),
      'meta'  => ['source'=>'company','company_id'=>$cid]
    ], JSON_UNESCAPED_UNICODE);
    exit;
  }

  // 4) Tag list only when list intent
  $tag = null;
  if (is_list_intent($q)) {
    $tag = detect_tag_in_question($pdo, $q);
  }
  if ($tag) {
    $intent = 'tag_search';
    $companies = find_by_tag($pdo, $tag);
    if ($companies) {
      $bullets = [];
      foreach($companies as $c){
        $booth = trim(($c['hall'] ?? '').' '.($c['booth'] ?? ''));
        $vip = !empty($c['is_vip']) ? ' (VIP)' : '';
        $bullets[] = $c['name'].$vip.($booth ? (" — ".$booth) : '');
        if (count($bullets) >= 10) break;
      }
      log_answer($pdo, $q, $intent, 'company', null);
      echo json_encode([
        'answer' => format_rich(
          'شرکت‌های مرتبط',
          'بر اساس Tag: '.$tag,
          $bullets,
          'اگر اسم یکی از شرکت‌ها رو بگی، جزئیات کاملش رو هم می‌دم.'
        ),
        'meta' => ['source'=>'company','tag'=>$tag]
      ], JSON_UNESCAPED_UNICODE);
      exit;
    }
  }
  // 5) Site data (skip for advice questions)
  if (!is_advice_intent($q)) {

  $hits = search_site($pdo, $q, 3);
  if ($hits) {
    log_answer($pdo, $q, 'site', 'site', null);
    [$title,$summary,$bullets,$extra] = build_site_answer($hits);
    echo json_encode([
      'answer' => format_rich($title, $summary, $bullets, $extra),
      'meta' => ['source'=>'site']
    ], JSON_UNESCAPED_UNICODE);
    exit;
  }
  }

  // 6) AI fallback
  $aiEnabled = setting_get($pdo,'ai_enabled','1') === '1';
  $apiKey = trim(setting_get($pdo,'openrouter_api_key',''));
  $model  = trim(setting_get($pdo,'openrouter_model','google/gemma-2-9b-it'));
  $sys    = trim(setting_get($pdo,'system_prompt',$app['response']['system_prompt'] ?? ''));
  $rules  = trim(setting_get($pdo,'response_rules',$app['response']['rules'] ?? ''));

  $sysFull = $sys."\n\nلحن: نیمه‌رسمی و حرفه‌ای مثل چت‌بات‌های نمایشگاهی. پاسخ‌ها فارسی و مرتبط با صنعت نفت ایران.\n"
            ."قالب پاسخ:\n".$rules;

  $context = "یادآوری: اگر داده کافی نداری، شفاف بگو و 1-2 سوال کوتاه برای دقیق‌تر شدن بپرس.\n";

  if ($aiEnabled && $apiKey) {
    $ai = openrouter_chat($q, $sysFull, $apiKey, $model, $context);
    if ($ai) {
      log_answer($pdo, $q, 'ai', 'ai', null);
      echo json_encode([
        'answer' => format_rich('پاسخ هوشمند', 'بر اساس سوال شما:', [trim($ai)], ''),
        'meta' => ['source'=>'ai','model'=>$model]
      ], JSON_UNESCAPED_UNICODE);
      exit;
    }
  }

  log_answer($pdo, $q, 'no_answer', 'none', null);
  echo json_encode([
    'answer' => format_rich(
      'اطلاعات کافی پیدا نشد',
      'فعلاً مورد دقیقی در دیتابیس/سایت پیدا نکردم.',
      ['اگر اسم دقیق شرکت یا یک کلمه کلیدی کوتاه (مثل: حفاری / HSE / پتروشیمی) رو بگی، سریع‌تر می‌تونم پیدا کنم.'],
      ''
    ),
    'meta' => ['source'=>'none']
  ], JSON_UNESCAPED_UNICODE);

} catch (Throwable $e) {
  // Always return JSON (helps debug HTTP 500)
  http_response_code(500);
  $msg = $debug ? ($e->getMessage().' @ '.$e->getFile().':'.$e->getLine()) : 'Internal server error';
  // Log to PHP error log too
  error_log('[oilshow-chat] '.$e->getMessage().' @ '.$e->getFile().':'.$e->getLine());
  echo json_encode([
    'error' => 'server_error',
    'message' => $msg
  ], JSON_UNESCAPED_UNICODE);
}
